# Agently Agent Development Guide (for Coding Agents)

Purpose: Help coding agents build **business-facing** agent systems with Agently. This is not for framework development.

## 0. Audience & Scope
- You are implementing **business workflows** on top of Agently.
- Use this guide to map **scenario → capability → concrete operations**.
- Keep outputs stable, testable, and reproducible.

## 1. Agent workflow (must follow)
1) **Problem mapping**: identify the scenario and list atomic goals.
2) **Capability selection**: Output control, Session/Memo, TriggerFlow, Streaming.
3) **Plan**: propose Spec → Design → Plan → Dev with acceptance tests.
4) **Branch**: if git is available, create a non‑disruptive branch (e.g., `dev-test-commit`).
5) **Batch delivery**: split tasks into small batches; each batch must include tests + expected outputs.
6) **Verify**: run tests and record results before reporting completion.

## 2. Model configuration
Local testing (Ollama):
```python
from agently import Agently

Agently.set_settings(
  "OpenAICompatible",
  {"base_url": "http://localhost:11434/v1", "model": "qwen2.5:7b", "model_type": "chat"},
).set_settings("request_options", {"temperature": 0.2}).set_settings("debug", False)
```

Cloud model (recommended for complex tasks): `model_configs/DeepSeek.py`.

## 3. Output control + streaming (critical)
Always define `output()` before structured streaming.

Correct pattern:
```python
response = (
  agent
  .input("...")
  .output({"summary": (str,), "items": [(str,)]})
  .get_response()  # fix a single request
)

# Text stream
for chunk in response.get_generator(type="delta"):
  ...

# Structured stream
for item in response.get_generator(type="instant"):
  if item.is_complete:
    ...

# Stable data
result = response.get_data(ensure_keys=["summary", "items[*]"])
```

Common mistakes:
- Calling `instant` without `output()`.
- Calling `get_text()` / `get_data()` repeatedly (causes multiple requests).
- Skipping `ensure_keys` for critical fields.

## 4. TriggerFlow (beyond when)
Use `to` as the main chain. Use `when` only for signal subscriptions.

Common patterns:
- **Condition**: `if_condition / elif_condition / else_condition`
- **Routing**: `match / case / case_else`
- **Batch**: `batch(..., concurrency=N)`
- **List**: `for_each(...)`
- **Signals**: `when({"runtime_data": "key"})`

Migration hints:
- Dify/LangGraph node → `chunk`
- Edge → `to`
- Router → `match` / `if_condition`
- State → `runtime_data`
- Parallel subgraph → `batch` / `for_each`

## 5. Tests & outputs (required for every batch)
For your **target project**, do not just run examples. Create your own tests:
- **Inputs**: define representative + edge case inputs.
- **Outputs**: define required schema + key presence (use `ensure_keys`).
- **Streaming**: verify `delta` + `instant` events are consumed correctly.
- **Memory**: verify memo updates and injection across turns.
- **TriggerFlow**: verify branches, concurrency limits, and aggregation results.

Record outputs under `outputs/` for reproducibility.

## 6. References (internal pack)
- `docs/agent-systems/overview.md`
- `docs/agent-docs.md`
- `docs/output-control/overview.md`
- `docs/agent-extensions/session-memo/quickstart.md`
- `docs/triggerflow/overview.md`
- `skills/` (skill modules)
- `examples/` + `outputs/`
