---
title: Agent Systems Playbook
---

# Agent Systems Playbook (Agently)

For **business developers** building agent systems with Agently. The goal is to map **Scenario → Capability (key traits) → Concrete Operations**, with **dedicated doc pages** that include code and real outputs.

> Compatibility: Python 3.10+, Agently 4.0.7.2+.

---

## Reading paths

- **Quick start**: sections 1–2.
- **Capability corrections**: section 3.
- **Playbook chapter**: section 4.

---

## 1. Problem‑solving workflow (for business builders)

**Recommended path: Scenario → Atomic goals → Capability selection → Concrete operations → Validation**

1) **Scenario**: inputs, outputs, business goal.
2) **Atomic goals**: intent routing / extraction / planning / routing / memory / orchestration / streaming.
3) **Capability selection**: Output control + ensure_keys / Session & Memo / TriggerFlow / Streaming.
4) **Operations**: the concrete Agently chain and code.
5) **Validation**: schema, key presence, streaming behavior, branch correctness.

If you need a more engineering workflow, use **Spec → Design → Plan → Dev** (define outputs and acceptance tests first).

---

## 2. Scenario → Capability → Operation (cheat sheet)

| Scenario | Capability (key traits) | Operations | Notes |
| --- | --- | --- | --- |
| Intent routing | Structured output (stable fields) | `output()` + `ensure_keys` | Plug into TriggerFlow branches |
| Info extraction | Structured output (lists/nesting) | `output()` + `ensure_keys` | Supports `[*]` paths |
| Planning/steps | Output order control | `output()` in dependency order | Facts first, summary last |
| Text streaming | Text stream | `get_generator(type="delta")` | Great for UI |
| Structured streaming | Instant structured stream | `get_generator(type="instant")` | Requires `output()` |
| Field‑level triggers | KeyWaiter | `when_key` / `wait_keys` | Early field handling |
| Multi‑turn memory | Session & Memo | `enable_session_memo()` + `info(memo)` | Memo must be injected |
| Orchestration | TriggerFlow | `to` / `if` / `match` / `for_each` / `batch` | Don’t overuse `when` |
| Dify/LangGraph migration | Graph mapping | chunk=node, to=edge, match/if=router | Fast logic migration |

---

## 3. Common pitfalls (critical corrections)

### 3.1 Output control + streaming (correct usage)

```python
response = (
  agent
  .input("...")
  .output({"summary": (str,), "items": [(str,)]})
  .get_response()  # fix one request
)

# Text stream
for chunk in response.get_generator(type="delta"):
  ...

# Structured stream
for item in response.get_generator(type="instant"):
  if item.is_complete:
    ...

# Stable data
result = response.get_data(ensure_keys=["summary", "items[*]"])
```

**Common mistakes**:
- Using `instant` without `output()`.
- Calling `get_text()` / `get_data()` multiple times in a chain (triggers multiple requests).
- Skipping `ensure_keys` for critical fields.

---

### 3.2 TriggerFlow is not just `when`

Typical combinations:
- **Main chain**: `to`
- **Conditions**: `if_condition / elif_condition / else_condition`
- **Routing**: `match / case / case_else` (hit_first / hit_all)
- **List processing**: `for_each`
- **Parallel**: `batch` + `concurrency`
- **Signals**: `when(event / runtime_data / flow_data)`

Migration hints (Dify / LangGraph):
- Node → `chunk`
- Edge → `to`
- Router → `match` / `if_condition`
- State → `runtime_data`
- Parallel subgraph → `batch` / `for_each`

---

## 4. Playbook chapter (docs with code + outputs)

Each scenario below has a **dedicated doc page** with full code and real outputs:

- Ticket triage: `/en/agent-systems/ticket-triage`
- Streaming + structured: `/en/agent-systems/streaming-structured`
- KeyWaiter early fields: `/en/agent-systems/key-waiter`
- Session & Memo: `/en/agent-systems/session-memo`
- TriggerFlow orchestration: `/en/agent-systems/triggerflow-orchestration`

---

## 5. Model configuration

- Common model settings: `docs/model-settings.md`
- Model docs: `docs/models/deepseek.md`, `docs/models/qwen.md`, etc.
- Local small models are good for tests; large cloud models are better for complex tasks.

---

## 6. Summary

- **Scenario → Capability → Operation** is the core learning path.
- Correct output control = `output()` + `instant` + `ensure_keys`.
- TriggerFlow’s common combo is `to + if/match + for_each + batch` (not only `when`).

For agent‑oriented instructions, see: `/en/agent-docs`.
