---
title: Session & Memo Playbook
---

# Session & Memo Playbook

## Scenario
Long conversations require **stable memory** and **context summarization**.

## Capability (key traits)
- **Session & Memo** for memory + summarization
- **Explicit memo injection** (memo is not auto‑injected)

## Operations
1) Enable memo mode.
2) Inject memo before each turn.

## Full code

```python
import json
from agently import Agently

Agently.set_settings(
    "OpenAICompatible",
    {
        "base_url": "http://localhost:11434/v1",
        "model": "qwen2.5:7b",
        "model_type": "chat",
    },
).set_settings("request_options", {"temperature": 0.2}).set_settings("debug", False)

agent = Agently.create_agent()
agent.enable_session_memo(chars=800, messages=6, every_n_turns=1)

agent.system("你是产品顾问，回答简短，并记住用户偏好。")

reply_1 = agent.input("我叫小航，正在做一款 AI 写作工具，目标用户是学生。给我 2 条建议。").get_text()
print("TURN1_REPLY:", reply_1)
print("TURN1_MEMO:", json.dumps(agent.session.memo, ensure_ascii=False, indent=2))

agent.info({"memo": agent.session.memo})
reply_2 = agent.input("我们预算不高，应该先做哪一个功能？").get_text()
print("TURN2_REPLY:", reply_2)
print("TURN2_MEMO:", json.dumps(agent.session.memo, ensure_ascii=False, indent=2))
```

## Real output

```text
TURN1_REPLY: 1. 提供多种文体和格式选择。
2. 加入范文学习功能帮助提高写作能力。
TURN1_MEMO: {
  "user_profile": {
    "name": "小航",
    "project_name": "AI 写作工具",
    "target_users": [
      "学生"
    ]
  },
  "suggestions": [
    "提供多种文体和格式选择。",
    "加入范文学习功能帮助提高写作能力。"
  ],
  "last_resize": {
    "type": "lite",
    "turn": 1,
    "reason": "every_n_turns"
  }
}
TURN2_REPLY: 在预算有限的情况下选择优先级较高的功能可以帮助你们更高效地推进项目。根据目标用户是学生这一特点，我建议首先实现“加入范文学习功能帮助提高写作能力”这个功能。

### 原因如下：
1. **直接提升写作文本质量**：对于学生而言，通过阅读和分析高质量的范文可以快速提升他们的写作水平和理解力，这对他们的长期发展非常有益。
2. **易于用户上手**：这种功能可以帮助新用户快速看到工具的实际效果，并产生使用兴趣，从而增加用户的黏性。
3. **逐步完善其他功能**：在积累了一定数量的用户反馈后，针对不同的文体需求提供丰富的模板和格式选择会更加方便。

当然，在实施这一功能的同时，也不宜忽视“提供多种文体和格式选择”的长远规划，可以适当简化初期版本的功能实现方式，逐步丰富功能。
TURN2_MEMO: {
  "last_resize": {
    "type": "lite",
    "turn": 2,
    "reason": "every_n_turns"
  },
  "suggestions": [
    "提供多种文体和格式选择。",
    "加入范文学习功能帮助提高写作能力。"
  ],
  "user_profile": {
    "name": "小航",
    "project_name": "AI 写作工具",
    "target_users": [
      "学生"
    ]
  },
  "priority_features": [
    {
      "feature": "加入范文学习功能帮助提高写作能力",
      "rationale": [
        "直接提升写作文本质量。",
        "易于用户上手，增加用户的黏性。",
        "逐步完善其他功能。"
      ],
      "action_plan": "在积累了一定数量的用户反馈后，针对不同的文体需求提供丰富的模板和格式选择。初期版本的功能实现方式可以适当简化。"
    },
    {
      "feature": "提供多种文体和格式选择",
      "rationale": [
        "满足多样化的写作需求。"
      ],
      "status": "长期规划"
    }
  ]
}
```

## Validation
- Memo updates after each turn.
- Memo affects the next response when injected.
