---
name: agently-agentic-workflow
description: Build business-facing agent systems in Agently (output control + streaming + session/memo + TriggerFlow). Use for multi-step workflows and agentic automation.
---

# Agently Agentic Workflow Skill

Use this skill to deliver **business workflows** with Agently. Focus on mapping **scenario → capability → concrete operations**.

## Required workflow (agent)
1) Identify scenario and atomic goals.
2) Pick capabilities (Output control / Session & Memo / TriggerFlow / Streaming).
3) Plan Spec → Design → Plan → Dev with acceptance tests.
4) Implement in batches; each batch must include tests + expected outputs.

## Required patterns
- **Output control**: always `output()` + `ensure_keys` for machine-consumed results.
- **Streaming**: `delta` for text, `instant` for structured streaming (same response).
- **Memory**: `enable_session_memo(...)`, inject memo each turn via `agent.info({"memo": ...})`.
- **Orchestration**: `to + if/match + for_each + batch` (not only `when`).

## Minimal skeleton
```python
from agently import Agently, TriggerFlow, TriggerFlowEventData

Agently.set_settings(
  "OpenAICompatible",
  {"base_url": "http://localhost:11434/v1", "model": "qwen2.5:7b", "model_type": "chat"},
)

agent = Agently.create_agent()
agent.enable_session_memo(chars=800, messages=6, every_n_turns=1)

flow = TriggerFlow()

@flow.chunk
async def step(data: TriggerFlowEventData):
  return (
    agent.input(data.value)
    .output({"result": (str,)})
    .async_start()
  )

flow.to(step).end()
print(flow.start("run"))
```

## Tests (project-level, mandatory)
- Define input samples + edge cases.
- Verify required keys with `ensure_keys`.
- Validate `delta` + `instant` streams.
- Validate memo updates across turns.
- Validate TriggerFlow branches and concurrency.

## References
- `docs/agent-systems/overview.md`
- `docs/agent-docs.md`
- `docs/output-control/overview.md`
- `docs/agent-extensions/session-memo/quickstart.md`
- `docs/triggerflow/overview.md`
- `examples/` + `outputs/`
