---
name: agently-output-control
description: Stable structured outputs with Output Format, ensure_keys, ordering, and structured streaming. Use when outputs must be machine-consumable or streamed per field.
---

# Agently Output Control Skill

Use this skill for **stable schemas** and **correct streaming**.

## Core rules
- Always define `output()` before structured streaming.
- Use `ensure_keys` for critical fields.
- Order fields so dependencies appear earlier.
- Fix one request with `get_response()` if you need multi‑consumer streaming + final data.

## Patterns
### 1) Structured output + ensure_keys
```python
result = (
  agent
  .input("...")
  .output({"intent": (str,), "confidence": (float,)})
  .start(ensure_keys=["intent", "confidence"], max_retries=1)
)
```

### 2) Correct streaming + stable data (single response)
```python
response = (
  agent
  .input("...")
  .output({"summary": (str,), "items": [(str,)]})
  .get_response()
)

for chunk in response.get_generator(type="delta"):
  ...

for item in response.get_generator(type="instant"):
  if item.is_complete:
    ...

data = response.get_data(ensure_keys=["summary", "items[*]"])
```

### 3) KeyWaiter early field
```python
(
  agent.input("...")
  .output({"risk": (str,), "decision": (str,)})
  .when_key("risk", lambda v: print(v))
  .start_waiter()
)
```

## References
- `docs/output-control/overview.md`
- `docs/output-control/format.md`
- `docs/output-control/ensure-keys.md`
- `docs/model-response/streaming.md`
- `docs/agent-extensions/key-waiter.md`
